#!/usr/bin/env python3
import os
import lab
import sys
import pickle
import random
import unittest
from unittest.mock import MagicMock, patch, call


TEST_DIRECTORY = os.path.dirname(__file__)

WEATHER_PREFIX = "https://api.weather.gov/points/"
DISTANCE_PREFIX = "https://www.zipcodeapi.com/rest/"
WEATHER_CAMBRIDGE = "https://api.weather.gov/points/42.393246,-71.133833"
WEATHER_OTHER = "https://api.weather.gov/points/36.887214,-86.453985"


def make_effect(temp, temp_cambridge, distance):
    def request_effect(url):
        x = MagicMock()

        if url == WEATHER_CAMBRIDGE:
            x.json.return_value = {'properties': {'forecast':'cambridge_forecast_url'}}
        elif url[:len(WEATHER_PREFIX)] == WEATHER_PREFIX:
            x.json.return_value = {'properties': {'forecast':'forecast_url'}}
        elif url == "forecast_url":
            x.json.return_value = {'properties':{'periods':[{"number":1, "temperature":temp}, {"number":2, "temperature":120}]}}
        elif url == "cambridge_forecast_url":
            x.json.return_value = {'properties':{'periods':[{"number":1, "temperature":temp_cambridge}, {"number":2, "temperature":60}]}}
        elif url[:len(DISTANCE_PREFIX)] == DISTANCE_PREFIX:
            x.json.return_value = {"distance":distance}
        return x
    return request_effect

class TestShipping(unittest.TestCase):
    @patch('lab.requests')
    def test_01_requests(self, mock_request):
        # In this test, the max temp will be 80, and with a distance of 2009.656, the number of ice packs should be 20.09 rounded up to 21
        mock_request.get.side_effect = make_effect(80, 73, 2009.656)

        order = {"ID":"52", "Address":"100 Valley Drive", "City":"Helena", "State":"MT", "Zip Code":"59602", "Item":"tomatoes", "Amount":"17"}
        result = lab.get_ice_packs(order)
        mock_request.assert_has_calls([call.get("https://api.weather.gov/points/42.393246,-71.133833"), call.get('forecast_url'),
            call.get("https://api.weather.gov/points/46.707449,-111.957961"), call.get('cambridge_forecast_url')], any_order=True)

        self.assertEqual(result, 21)

    @patch('lab.requests')
    def test_02_requests(self, mock_request):
        # In this test, the max temp will be 65, and with a distance of 2009.656, the number of ice packs should be 0
        mock_request.get.side_effect = make_effect(60, 65, 2009.656)

        order = {"ID":"52", "Address":"100 Valley Drive", "City":"Helena", "State":"MT", "Zip Code":"59602", "Item":"tomatoes", "Amount":"17"}
        result = lab.get_ice_packs(order)
        mock_request.assert_has_calls([call.get("https://api.weather.gov/points/42.393246,-71.133833"), call.get('forecast_url'),
            call.get("https://api.weather.gov/points/46.707449,-111.957961"), call.get('cambridge_forecast_url')], any_order=True)
        self.assertEqual(result, 0)

    @patch('lab.requests')
    def test_03_requests(self, mock_request):
        # In this test, the max temp will be 90, and with a distance of 1000, the number of ice packs should be exactly 20
        mock_request.get.side_effect = make_effect(90, 90, 1000)
        order = {"ID":"5", "Address":"34 Main St", "City":"Parsons", "State":"WV", "Zip Code":"26287", "Item":"oranges", "Amount":"3"}
        result = lab.get_ice_packs(order)
        mock_request.assert_has_calls([call.get("https://api.weather.gov/points/39.096016,-79.678724"),
                                       call.get('forecast_url'), call.get("https://api.weather.gov/points/42.393246,-71.133833"),
                                       call.get('cambridge_forecast_url')], any_order=True)
        self.assertEqual(result, 20)

    @patch('lab.requests')
    def test_04_requests(self, mock_request):
        # In this test, the max temp will be 90, and with a distance of 2062.942, the number of ice packs should be 63.951 rounded to 64
        mock_request.get.side_effect = make_effect(101, 54, 2062.942)
        order = {"ID":"5", "Address":"500 Olive Avenue", "City":"El Paso", "State":"TX", "Zip Code":"79930", "Item":"umbrellas", "Amount":"32"}
        result = lab.get_ice_packs(order)
        mock_request.assert_has_calls([call.get("https://api.weather.gov/points/31.809457,-106.464242"), call.get('forecast_url'),
            call.get("https://api.weather.gov/points/42.393246,-71.133833"), call.get('cambridge_forecast_url')], any_order=True)
        self.assertEqual(result, 64)

class TestTemp(unittest.TestCase):
    @patch('lab.requests')
    def test_temp_cambridge(self, mock_request):
        # In this test, the max temp will be 80, and with a distance of 2009.656, the number of ice packs should be 20.09 rounded up to 21
        mock_request.get.side_effect = make_effect(80, 93, 2009.656)
        lat,long = 42.393246,-71.133833
        result = lab.get_temp(lat, long)
        mock_request.assert_has_calls([call.get(WEATHER_CAMBRIDGE), call.get('cambridge_forecast_url')])

        self.assertEqual(result, 93)

    @patch('lab.requests')
    def test_temp_random(self, mock_request):
        # In this test, the max temp will be 65, and with a distance of 2009.656, the number of ice packs should be 0
        temp = random.randint(80,100)
        mock_request.get.side_effect = make_effect(temp, 65, 2009.656)
        lat, long = 36.887214,-86.453985
        order = {"ID":"52", "Address":"100 Valley Drive", "City":"Helena", "State":"MT", "Zip Code":"59602", "Item":"tomatoes", "Amount":"17"}
        result = lab.get_temp(lat, long)
        #mock_request.get.assert_any_call(WEATHER_OTHER)
        #mock_request.get.assert_any_call('forecast_url')
        mock_request.assert_has_calls([call.get(WEATHER_OTHER), call.get('forecast_url'),])
        self.assertEqual(result, temp)

if __name__ == '__main__':
    res = unittest.main(verbosity=3, exit=False)
